/*
    FreeRTOS V9.0.0 - Copyright (C) 2016 Real Time Engineers Ltd.
    All rights reserved

    VISIT http://www.FreeRTOS.org TO ENSURE YOU ARE USING THE LATEST VERSION.

    This file is part of the FreeRTOS distribution.

    FreeRTOS is free software; you can redistribute it and/or modify it under
    the terms of the GNU General Public License (version 2) as published by the
    Free Software Foundation >>>> AND MODIFIED BY <<<< the FreeRTOS exception.

    ***************************************************************************
    >>!   NOTE: The modification to the GPL is included to allow you to     !<<
    >>!   distribute a combined work that includes FreeRTOS without being   !<<
    >>!   obliged to provide the source code for proprietary components     !<<
    >>!   outside of the FreeRTOS kernel.                                   !<<
    ***************************************************************************

    FreeRTOS is distributed in the hope that it will be useful, but WITHOUT ANY
    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
    FOR A PARTICULAR PURPOSE.  Full license text is available on the following
    link: http://www.freertos.org/a00114.html

    ***************************************************************************
     *                                                                       *
     *    FreeRTOS provides completely free yet professionally developed,    *
     *    robust, strictly quality controlled, supported, and cross          *
     *    platform software that is more than just the market leader, it     *
     *    is the industry's de facto standard.                               *
     *                                                                       *
     *    Help yourself get started quickly while simultaneously helping     *
     *    to support the FreeRTOS project by purchasing a FreeRTOS           *
     *    tutorial book, reference manual, or both:                          *
     *    http://www.FreeRTOS.org/Documentation                              *
     *                                                                       *
    ***************************************************************************

    http://www.FreeRTOS.org/FAQHelp.html - Having a problem?  Start by reading
    the FAQ page "My application does not run, what could be wrong?".  Have you
    defined configASSERT()?

    http://www.FreeRTOS.org/support - In return for receiving this top quality
    embedded software for free we request you assist our global community by
    participating in the support forum.

    http://www.FreeRTOS.org/training - Investing in training allows your team to
    be as productive as possible as early as possible.  Now you can receive
    FreeRTOS training directly from Richard Barry, CEO of Real Time Engineers
    Ltd, and the world's leading authority on the world's leading RTOS.

    http://www.FreeRTOS.org/plus - A selection of FreeRTOS ecosystem products,
    including FreeRTOS+Trace - an indispensable productivity tool, a DOS
    compatible FAT file system, and our tiny thread aware UDP/IP stack.

    http://www.FreeRTOS.org/labs - Where new FreeRTOS products go to incubate.
    Come and try FreeRTOS+TCP, our new open source TCP/IP stack for FreeRTOS.

    http://www.OpenRTOS.com - Real Time Engineers ltd. license FreeRTOS to High
    Integrity Systems ltd. to sell under the OpenRTOS brand.  Low cost OpenRTOS
    licenses offer ticketed support, indemnification and commercial middleware.

    http://www.SafeRTOS.com - High Integrity Systems also provide a safety
    engineered and independently SIL3 certified version for use in safety and
    mission critical applications that require provable dependability.

    1 tab == 4 spaces!
*/

/*-----------------------------------------------------------
 * Implementation of functions defined in portmacro.h for the NDS32 port.
 *----------------------------------------------------------*/

/* Scheduler includes. */
#include "FreeRTOS.h"
#include "task.h"
#include "ISR_Support.h"


#ifndef configMAX_API_CALL_INTERRUPT_PRIORITY
#define configMAX_API_CALL_INTERRUPT_PRIORITY
#error configMAX_API_CALL_INTERRUPT_PRIORITY must be defined.
#endif

#if( ( configMAX_API_CALL_INTERRUPT_PRIORITY > 3 ) || ( configMAX_API_CALL_INTERRUPT_PRIORITY < 0 ) )
#error configMAX_API_CALL_INTERRUPT_PRIORITY must be between 0 and 3
#endif

#ifndef configIRQ_SWI_VECTOR
#error configIRQ_SWI_VECTOR must be defined in FreeRTOSConfig.h to have SWI to perform context switch.
#endif

#ifndef configSETUP_TICK_INTERRUPT
#error configSETUP_TICK_INTERRUPT() must be defined in FreeRTOSConfig.h to call the function that sets up the tick interrupt.
#endif

#ifndef configCLEAR_TICK_INTERRUPT
#error configCLEAR_TICK_INTERRUPT must be defined in FreeRTOSConfig.h to clear which ever interrupt was used to generate the tick interrupt.
#endif


/* Constants required to set up the initial stack. */
#define portPSW_GIE         ( 1UL << 0UL )
#define portPSW_AEN         ( 1UL << 13UL )
#define portPSW_IFCON           ( 1UL << 15UL )
#define portPSW_CPL         ( 7UL << 16UL )

#if ( configSUPPORT_ZOL == 1 )
/* Default task PSW: enable GIE and AEN, set CPL to 7, clear IFCON */
#define portINITIAL_PSW         ( ( __nds32__mfsr( NDS32_SR_PSW ) | portPSW_GIE | portPSW_CPL | portPSW_AEN ) & ~portPSW_IFCON )
#else
/* Default task PSW: enable GIE, set CPL to 7, clear IFCON */
#define portINITIAL_PSW         ( ( __nds32__mfsr( NDS32_SR_PSW ) | portPSW_GIE | portPSW_CPL ) & ~portPSW_IFCON )
#endif


/* Each task maintains its own interrupt status in the critical nesting
variable. */
volatile UBaseType_t uxCriticalNesting = 0xaaaaaaaa;

/*
 * Start first task is a separate function so it can be tested in isolation.
 */
extern void vPortStartFirstTask(void);

/*
 * Used to catch tasks that attempt to return from their implementing function.
 */
static void prvTaskExitError(void);

/*-----------------------------------------------------------*/

/*
 * Initialise the stack of a task to look exactly as if a call to
 * portSAVE_CONTEXT had been called.
 *
 * Stack Layout:
 *                High  |-----------------|
 *                      |       $R30 (LP) |
 *                      |-----------------|
 *                      |       $R29 (GP) |
 *                      |-----------------|
 *                      |       $R28 (FP) |
 *                      |-----------------|
 *                      |   $R15 | $R25   |
 *                      |-----------------|
 *                      |   $R10 | $R24   |
 *                      |-----------------|
 *                      |        .        |
 *                      |        .        |
 *                      |-----------------|
 *                      |       $R0       |
 *                      |-----------------|
 *                      |       $IFC_LP   | ( configSUPPORT_IFC )
 *                      |-----------------|
 *                      |   $LC/$LE/$LB   | ( configSUPPORT_ZOL )
 *                      |       (ZOL)     |
 *                      |-----------------|
 *                      |       $IPSW     |
 *                      |-----------------|
 *                      |       $IPC      |
 *                      |-----------------|
 *                      |    Dummy word   | ( Dummy word for 8-byte stack pointer alignment )
 *                      |-----------------|
 *                      |       $FPU      | ( configSUPPORT_FPU )
 *                      |-----------------|
 *                Low
 *
 */
StackType_t *pxPortInitialiseStack(StackType_t *pxTopOfStack, TaskFunction_t pxCode, void *pvParameters)
{
    /* Simulate the stack frame as it would be created by a context switch
    interrupt. */
    extern uint32_t _SDA_BASE_ __attribute__((weak));
    /* R0 ~ R30 registers */
    *--pxTopOfStack = (StackType_t) prvTaskExitError;  /* R30 : $lp */
    *--pxTopOfStack = (StackType_t) &_SDA_BASE_;     /* R29 : $GP */
    *--pxTopOfStack = (StackType_t) 0x1c;        /* R28 */
#ifdef __NDS32_REDUCE_REGS__
    *--pxTopOfStack = (StackType_t) 0x0f;        /* R15 */
#else
    *--pxTopOfStack = (StackType_t) 0x19;        /* R25 */
    *--pxTopOfStack = (StackType_t) 0x18;        /* R24 */
    *--pxTopOfStack = (StackType_t) 0x17;        /* R23 */
    *--pxTopOfStack = (StackType_t) 0x16;        /* R22 */
    *--pxTopOfStack = (StackType_t) 0x15;        /* R21 */
    *--pxTopOfStack = (StackType_t) 0x14;        /* R20 */
    *--pxTopOfStack = (StackType_t) 0x13;        /* R19 */
    *--pxTopOfStack = (StackType_t) 0x12;        /* R18 */
    *--pxTopOfStack = (StackType_t) 0x11;        /* R17 */
    *--pxTopOfStack = (StackType_t) 0x10;        /* R16 */
    *--pxTopOfStack = (StackType_t) 0x0f;        /* R15 */
    *--pxTopOfStack = (StackType_t) 0x0e;        /* R14 */
    *--pxTopOfStack = (StackType_t) 0x0d;        /* R13 */
    *--pxTopOfStack = (StackType_t) 0x0c;        /* R12 */
    *--pxTopOfStack = (StackType_t) 0x0b;        /* R11 */
#endif
    *--pxTopOfStack = (StackType_t) 0x0a;        /* R10 */
    *--pxTopOfStack = (StackType_t) 0x09;           /* R9 */
    *--pxTopOfStack = (StackType_t) 0x08;        /* R8 */
    *--pxTopOfStack = (StackType_t) 0x07;        /* R7 */
    *--pxTopOfStack = (StackType_t) 0x06;        /* R6 */
    *--pxTopOfStack = (StackType_t) 0x05;        /* R5 */
    *--pxTopOfStack = (StackType_t) 0x04;        /* R4 */
    *--pxTopOfStack = (StackType_t) 0x03;        /* R3 */
    *--pxTopOfStack = (StackType_t) 0x02;        /* R2 */
    *--pxTopOfStack = (StackType_t) 0x01;        /* R1 */
    *--pxTopOfStack = (StackType_t) pvParameters;    /* R0 : Argument */
#if ( configSUPPORT_IFC == 1 )
    /* IFC system register */
    *--pxTopOfStack = (StackType_t) 0x00;        /* IFC_LP */
#endif
#if ( configSUPPORT_ZOL == 1 )
    /* ZOL system registers */
    *--pxTopOfStack = (StackType_t) 0x00;        /* LC */
    *--pxTopOfStack = (StackType_t) 0x00;        /* LE */
    *--pxTopOfStack = (StackType_t) 0x00;        /* LB */
#endif
    /* IPSW and IPC system registers */
    *--pxTopOfStack = (StackType_t) portINITIAL_PSW;     /* IPSW */
    *--pxTopOfStack = (StackType_t) pxCode;          /* IPC : First instruction PC of task */
#if ( ( configSUPPORT_IFC == 1 ) && ( configSUPPORT_ZOL == 1 ) || ( configSUPPORT_IFC != 1 ) && ( configSUPPORT_ZOL != 1 ) )
    /* Dummy word for 8-byte stack alignment */
    *--pxTopOfStack = (portSTACK_TYPE) - 1;
#endif
#if ( configSUPPORT_FPU == 1 )
    /* FPU registers */
    pxTopOfStack -= portFPU_REGS;
#endif
    return pxTopOfStack;
}
/*-----------------------------------------------------------*/

static void prvTaskExitError(void)
{
    /* A function that implements a task must not exit or attempt to return to
    its caller as there is nothing to return to.  If a task wants to exit it
    should instead call vTaskDelete( NULL ).

    Artificially force an assert() to be triggered if configASSERT() is
    defined, then stop here so application writers can catch the error. */
    configASSERT(uxCriticalNesting == ~0UL);
    portDISABLE_INTERRUPTS();

    for (;;);
}
/*-----------------------------------------------------------*/

/*
 * See header file for description.
 */
BaseType_t xPortStartScheduler(void)
{
    /* Make Software interrupt the lowest priority interrupts. */
#if ( configIRQ_SWI_VECTOR < 16 )
    __nds32__mtsr(__nds32__mfsr(NDS32_SR_INT_PRI) | (3 << ((configIRQ_SWI_VECTOR) << 1)), NDS32_SR_INT_PRI);
#else
    __nds32__mtsr(__nds32__mfsr(NDS32_SR_INT_PRI2) | (3 << (((configIRQ_SWI_VECTOR) - 16) << 1)), NDS32_SR_INT_PRI2);
#endif
    __nds32__mtsr(__nds32__mfsr(NDS32_SR_INT_MASK2) | (1 << (configIRQ_SWI_VECTOR)), NDS32_SR_INT_MASK2);
    /* Start the timer that generates the tick ISR.  Interrupts are disabled
    here already. */
    /* Call an application function to set up the timer that will generate the
    tick interrupt.  This way the application can decide which peripheral to
    use.  A demo application is provided to show a suitable example. */
    configSETUP_TICK_INTERRUPT();
    /* Initialise the critical nesting count ready for the first task. */
    uxCriticalNesting = 0;
    /* Start the first task. */
    vPortStartFirstTask();
    /* Should not get here! */
    return 0;
}
/*-----------------------------------------------------------*/

void vPortEndScheduler(void)
{
    /* Not implemented in ports where there is nothing to return to.
    Artificially force an assert. */
    configASSERT(uxCriticalNesting == 1000UL);
}
/*-----------------------------------------------------------*/

void vPortEnterCritical(void)
{
    portDISABLE_INTERRUPTS();
    uxCriticalNesting++;
}
/*-----------------------------------------------------------*/

void vPortExitCritical(void)
{
    configASSERT(uxCriticalNesting);
    uxCriticalNesting--;

    if (uxCriticalNesting == 0) {
        portENABLE_INTERRUPTS();
    }
}
/*-----------------------------------------------------------*/

void FreeRTOS_Tick_Handler(void)
{
    /* The SysTick runs at the lowest interrupt priority, so when this interrupt
    executes all interrupts must be unmasked. */
    uint32_t ulPreviousMask;
    configCLEAR_TICK_INTERRUPT();
    ulPreviousMask = portSET_INTERRUPT_MASK_FROM_ISR();
    {
        /* Increment the RTOS tick. */
        if (xTaskIncrementTick() != pdFALSE) {
            /* A context switch is required.  Context switching is performed in
            the SWI interrupt.  Pend the SWI interrupt. */
            portYIELD();
        }
    }
    portCLEAR_INTERRUPT_MASK_FROM_ISR(ulPreviousMask);
}
/*-----------------------------------------------------------*/

#if( configASSERT_DEFINED == 1 )

void vPortValidateInterruptPriority(void)
{
#if ( configMAX_API_CALL_INTERRUPT_PRIORITY != 0 )
    uint8_t ucCurrentPriority;
    /* Obtain the number of the currently executing interrupt. */
    ucCurrentPriority = ((__nds32__mfsr(NDS32_SR_PSW) & portPSW_CPL_MASK) >> 16UL);
    /* The following assertion will fail if a service routine (ISR) for
    an interrupt that has been assigned a priority above
    configMAX_SYSCALL_INTERRUPT_PRIORITY calls an ISR safe FreeRTOS API
    function.  ISR safe FreeRTOS API functions must *only* be called
    from interrupts that have been assigned a priority at or below
    configMAX_SYSCALL_INTERRUPT_PRIORITY.

    Numerically low interrupt priority numbers represent logically high
    interrupt priorities, therefore the priority of the interrupt must
    be set to a value equal to or numerically *higher* than
    configMAX_SYSCALL_INTERRUPT_PRIORITY.

    FreeRTOS maintains separate thread and ISR API functions to ensure
    interrupt entry is as fast and simple as possible. */
    configASSERT(ucCurrentPriority >= configMAX_API_CALL_INTERRUPT_PRIORITY);
#endif
}

#endif /* configASSERT_DEFINED */
